"use strict";
/**
 * Markdown 格式化工具
 * 使用 Lexical Headless Editor 在 Node.js 环境中进行格式化
 * 统一的格式化规则，用于：
 * 1. exportMarkdown (webview 导出)
 * 2. formatMarkdown (extension 命令)
 * 3. normalizeMarkdown (diff 计算)
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatMarkdown = formatMarkdown;
require("prismjs");
require("prismjs/components/prism-mermaid.min.js");
const headless_1 = require("@lexical/headless");
const markdown_1 = require("@lexical/markdown");
const rich_text_1 = require("@lexical/rich-text");
const list_1 = require("@lexical/list");
const code_1 = require("@lexical/code");
const table_1 = require("@lexical/table");
const link_1 = require("@lexical/link");
const transformers_1 = require("./transformers");
/**
 * 格式化 Markdown 内容
 * 使用 Lexical Headless Editor 进行解析和序列化，确保格式一致性
 *
 * @param markdown 原始 Markdown 内容
 * @returns 格式化后的 Markdown 内容
 */
async function formatMarkdown(markdown) {
    if (!markdown || markdown.trim() === '') {
        return '';
    }
    try {
        // 【编码检测】验证输入内容的编码完整性
        const inputSize = markdown.length;
        const inputChars = markdown.match(/[\u4e00-\u9fff]/g)?.length ?? 0; // 中文字符计数
        if (inputChars > 0) {
            console.log(`[Markdown Formatter] Input contains ${inputChars} CJK characters (total: ${inputSize} chars)`);
        }
        // 创建 headless editor（无需 DOM）
        const editor = (0, headless_1.createHeadlessEditor)({
            nodes: [
                rich_text_1.HeadingNode,
                rich_text_1.QuoteNode,
                list_1.ListItemNode,
                list_1.ListNode,
                code_1.CodeNode,
                table_1.TableNode,
                table_1.TableCellNode,
                table_1.TableRowNode,
                link_1.LinkNode,
                // 注意：暂时不包含自定义 Node（TodoList, Mermaid），因为它们依赖浏览器 API
                // 如果需要支持，需要创建 Node.js 兼容版本
            ],
            onError: (error) => {
                console.error('[Markdown Formatter] Lexical error:', error);
            },
        });
        let formatted = '';
        // 使用 editor.update 来解析和序列化
        // 注意：Transformer 顺序很重要！
        // 参考 webview 的顺序：MERMAID, TABLE, ...TRANSFORMERS, TODO_LIST
        const allTransformers = [
            transformers_1.MERMAID_TRANSFORMER,
            transformers_1.TABLE_TRANSFORMER,
            ...markdown_1.TRANSFORMERS,
            transformers_1.TODO_LIST_TRANSFORMER,
        ];
        // 在同一个 update 中完成解析和序列化，避免时序问题
        editor.update(() => {
            // 1. 从 Markdown 字符串解析到 EditorState
            (0, markdown_1.$convertFromMarkdownString)(markdown, allTransformers);
            // 2. 立即在同一个事务中读取并序列化
            // 注意：这里不能用 read()，因为我们在 update 事务中
            formatted = (0, markdown_1.$convertToMarkdownString)(allTransformers);
        });
        // 基本的后处理：移除文件末尾多余空行，确保只有一个换行符
        formatted = formatted.trim();
        // 【编码验证】输出内容的编码完整性检查
        const outputSize = formatted.length;
        const outputChars = formatted.match(/[\u4e00-\u9fff]/g)?.length ?? 0;
        if (outputChars > 0) {
            console.log(`[Markdown Formatter] Output contains ${outputChars} CJK characters (total: ${outputSize} chars)`);
            // 警告：如果中文字符数量大幅下降，说明可能有损失
            if (inputChars > 0 && outputChars < inputChars * 0.9) {
                console.warn(`[Markdown Formatter] ⚠️  CJK character count decreased: ${inputChars} → ${outputChars}`);
            }
        }
        return formatted;
    }
    catch (error) {
        console.error('[Markdown Formatter] Failed to format markdown:', error);
        // 失败时返回原内容
        return markdown;
    }
}
//# sourceMappingURL=markdownFormatter.js.map